/* Condorcet vote expander

   Copyright 2006 Alexandre Oliva <lxoliva@fsfla.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street - Fifth Floor, Boston, MA 02110-1301, USA.  */

/* Given in stdin a list of votes of the form n1 {[>=] n2}*, where n
   is an integral number from -1 to 27, it verifies that no duplicates
   are present, and expands the votes in stdout such that every vote
   mentions every option exactly once, according to the special
   meaning of options -1 and 0.  0 stands for all positive options not
   mentioned explicitly, with equal preference.  If 0 is absent, it's
   given lower priority than all the mentioned options.  If -1 is
   absent, it's given lower priority than all other options.  */

#include <iostream>
#include <vector>
#include <list>

const int positive_options = 27;
const int negative_options = 1;

static bool
add_vote (int option, std::list<int>& current, std::vector<bool>& chosen)
{
  /* Reject out-of-range options.  */
  if (option < -negative_options || option > positive_options)
    {
      std::cerr << "Invalid option " << option << std::endl;
      return false;
    }

  /* Reject duplicates.  */
  if (chosen[option + negative_options])
    {
      std::cerr << "Duplicate option " << option << std::endl;
      return false;
    }

  /* Add the option to the current list and mark it as referenced.  */
  current.push_back (option);
  chosen[option + negative_options] = true;

  return true;
}

int main() {
  for (int option; std::cin >> option; ) {
    /* This is a list of lists, such that members of an inner list
       have the same preference, and inner lists are in decreasing
       order of preference.  */
    std::list<std::list<int> > vote;
    std::vector<bool> chosen(negative_options + positive_options);

    /* Since we got an option above, create the inner list for it and
       add the option to it.  */
    vote.push_back (std::list<int> ());

    if (!add_vote (option, vote.back (), chosen))
      return -1;

    char separator;
    if (std::cin >> separator)
      for (;;)
	{
	  /* Process all equal-preference members.  */
	  while (separator == '=')
	    {
	      if (!(std::cin >> option))
		{
		  std::cerr << "missing option" << std::endl;
		  return -1;
		}
	      if (!add_vote (option, vote.back (), chosen))
		return -1;
	      if (!(std::cin >> separator))
		separator = EOF;
	    }

	  if (separator == '>')
	    {
	      /* Start a new, lower priority list.  */
	      vote.push_back (std::list<int> ());
	      /* Enter the loop above in the next iteration.  */
	      separator = '=';
	    }
	  else
	    {
	      /* We're done with this vote.  Restore the next character.  */
	      std::cin.unget();
	      break;
	    }
	}

    /* If we didn't get a 0 vote, add one.  */
    if (!chosen[0+negative_options])
      {
	vote.push_back (std::list<int> ());
	add_vote (0, vote.back (), chosen);
      }

    /* If we didn't get a -1 vote, add one.  */
    if (!chosen[-1+negative_options])
      {
	vote.push_back (std::list<int> ());
	add_vote (-1, vote.back (), chosen);
      }

    /* Decide whether 0 expands to anything.  */
    bool empty_zero = true;
    for (option = 1; option <= positive_options; option++)
      {
	if (!chosen[option+negative_options])
	  empty_zero = false;
      }

    /* Now look for zero and either remove it, if it expands to
       nothing, or replace it with its expansion otherwise.  */
    for (std::list<std::list<int> >::iterator itx = vote.begin (),
	   ex = vote.end (); itx != ex; ++itx)
      for (std::list<int>::iterator it = itx->begin (),
	     e = itx->end (); it != e; ++it)
	if (*it == 0)
	  {
	    if (empty_zero)
	      {
		it = itx->erase (it);

		if (itx->empty ())
		  itx = vote.erase (itx);
	      }
	    else
	      {
		std::insert_iterator<std::list<int> > inserter(*itx, it);

		for (option = 1; option <= positive_options; option++)
		  if (!chosen[option+negative_options])
		    inserter = option;

		it = itx->erase (it);
	      }
	  }

    /* Display the vote.  */
    for (std::list<std::list<int> >::iterator itx = vote.begin (),
	   ex = vote.end (); itx != ex;)
      {
	/* Display votes with the same priority, separated by '='.  */
	for (std::list<int>::iterator it = itx->begin (),
	       e = itx->end (); it != e;)
	  {
	    std::cout << *it;
	    if (++it != e)
	      std::cout << '=';
	  }

	/* Display the '>' separator if we're not at the end.  */
	if (++itx != ex)
	  std::cout << '>';
      }

    std::cout << std::endl;
  }
}
